﻿using Spectral1.DATA_ACCESS;
using Spectral1_VBClassLibrary;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using static Spectral1.DATA_ACCESS.DA_3DWireframe;
using static Spectral1.DATA_ACCESS.DA_Spectral;
using static Spectral1_VBClassLibrary.DataSet_Spectral;

namespace Spectral1.BUSINESS_LOGIC
{
    public class c_waveform
    {
        #region "================ DECLARATIONS ========================"

        static CodeGen_DS_Spectral _CGS;
        static DA_Spectral _DASP;

        private c_harmonic[] harmonic = new c_harmonic[DA_Spectral.max_harmonics];

        private Int32 _waveform_set_id = 0;
        private Int32 _note_sector_id = 0;
        private Int32 _intensity_layer_id = 0;
        private Int32 _waveform_id = 0;
        private double last_calculated_level_scaling = 1;
        private bool flag_level_scaling_recalc_due = true;
        #endregion

        #region "================ PROPERTIES ========================"
        public Int32 waveform_set_id { get { return _waveform_set_id; } }

        public Int32 note_sector_id { get { return _note_sector_id; } }

        public Int32 intensity_layer_id { get { return _intensity_layer_id; } }

        public Int32 waveform_id { get { return _waveform_id; } }

        private waveformRow r
        {
            get
            {
                return _CGS.Table_waveform.GetRow(waveform_set_id, note_sector_id, intensity_layer_id, waveform_id);
            }
        }

        public string description
        {
            get
            {
                if (r == null)
                {
                    return "";
                }
                else
                {
                    return r.description;
                }
            }
            set
            {
                if (r != null)
                { r.description = value; }
            }

        }
        #endregion

        #region "================ METHODS ========================"
        public void clear_all_harmonic_levels()
        {
            for (int h = 0; h < max_harmonics; h++)
            {
                set_harmonic_level(h, 0);
            }
        }

        public void clear_all_harmonic_phases()
        {
            for (int h = 0; h < max_harmonics; h++)
            {
                set_harmonic_phase(h, true);
            }
        }

        public c_waveform(CodeGen_DS_Spectral CGS, DA_Spectral DASP,Int32 ws, Int32 s, Int32 i, Int32 w)
        {
            _CGS = CGS;
            _DASP = DASP;
            _waveform_set_id = ws;
            _note_sector_id = s;
            _intensity_layer_id = i;
            _waveform_id = w;

            for (Int32 h = 0; h < DA_Spectral.max_harmonics; h++)
            {
                harmonic[h] = new BUSINESS_LOGIC.c_harmonic(_CGS, ws, s, i, w, h);
            }
        }

        public Int32 get_harmonic_level(Int32 h) { return harmonic[h].level; }

        public void clear_harmonic(Int32 h) { harmonic[h].clear(); set_harmonics_changed(); }

        public void scale_harmonic_level(Int32 h, double factor) { set_harmonic_level(h, (Int32)(get_harmonic_level(h) * factor)); set_harmonics_changed(); }

        public bool get_harmonic_phase(Int32 h) { return harmonic[h].phase; }

        public phases get_harmonic_phase_enum(Int32 h)
        {
            if (harmonic[h].phase)
            { return phases.Inverted; }
            else
            { return phases.Normal; }
        }

        public void set_harmonic_level(Int32 h, Int32 l)
        {
          harmonic[h].level = l;
          _CGS.Table_waveform_harmonic.GetRow(_waveform_set_id, _note_sector_id, _intensity_layer_id, _waveform_id, h).level = l;
          set_harmonics_changed();
          flag_level_scaling_recalc_due = true;
        }

        public void set_harmonic_phase(Int32 h, bool p)
        {
            harmonic[h].phase = p;
            if (p)
            {
                _CGS.Table_waveform_harmonic.GetRow(_waveform_set_id, _note_sector_id, _intensity_layer_id, _waveform_id, h).phase_offset = 1;
            }
            else
            {
                _CGS.Table_waveform_harmonic.GetRow(_waveform_set_id, _note_sector_id, _intensity_layer_id, _waveform_id, h).phase_offset = 0;
            }
            set_harmonics_changed();
        }

        public void set_harmonic_phases(bool p)
        {
            for (int h = 0; h < max_harmonics; h++)
            { set_harmonic_phase(h, p); }
        }


        public void set_harmonics_from_array(Int32[] input)
        {
            for (Int32 h = 0; h < DA_Spectral.max_harmonics; h++)
            {
                set_harmonic_level(h, input[h]);
            }
        }

        private void set_harmonics_changed()
        {
            _DASP.data_changed.waveset.set_waveform_harmonics_changed(note_sector_id, intensity_layer_id, waveform_id, true);
        }

        public void set_harmonics_from_waveform(c_waveform w)
        {
            for (Int32 h = 0; h < DA_Spectral.max_harmonics; h++)
            {
                set_harmonic_level(h, w.get_harmonic_level(h));
                set_harmonic_phase(h, w.harmonic[h].phase);
            }
        }

        public string get_csv_list_of_harmonics()
        {
            string csv_string = "";

            for (Int32 h = 0; h < DA_Spectral.max_harmonics; h++)
            {
                csv_string += get_harmonic_level(h).ToString();
                if (h < DA_Spectral.max_harmonics - 1) { csv_string += ","; }
            }
            return csv_string;
        }

        private string convert_harmonic_list_to_csv(List<string> L)
        {
            string csv = "";
            for (int i = 0; i < DA_Spectral.max_harmonics; i++)
            {
                csv += L[i];
                if (i < (DA_Spectral.max_harmonics - 1))
                { csv += ","; }
            }
            return csv;
        }

        public string set_harmonics_from_csv_list2(List<string> L)
        {
            return set_harmonics_from_csv_list(convert_harmonic_list_to_csv(L));
        }
        public string set_harmonics_from_csv_list(string input)
        {
            string error_text = "";
            try
            {
                string[] in_array;
                in_array = input.Split(',');
                if (in_array.Length != DA_Spectral.max_harmonics)
                {
                    error_text = "Import failed : Incorrect number of harmonics to import - there should be " + DA_Spectral.max_harmonics.ToString();
                }
                else
                {
                    for (Int32 h = 0; h < DA_Spectral.max_harmonics; h++)
                    {
                        if (BL_Spectral.IsNumeric(in_array[h]) == false)
                        {
                            error_text = "Import failed : Non-numeric character in import data";
                            break;
                        }

                        if (Convert.ToInt32(in_array[h]) < 0 || Convert.ToInt32(in_array[h]) > 65535)
                        {
                            error_text = "Import failed : Harmonic level to import is outside of range 0 to 65535 ";
                        }
                        set_harmonic_level(h, Convert.ToInt32(in_array[h]));
                    }
                }
            }
            catch
            {
                error_text = "Import failed : Error in import data";
            }
            return error_text;
        }

        public string set_harmonic_phases_from_list(List<string> L)
        {
            string error_text = "";

            try
            {
                for (Int32 h = 0; h < DA_Spectral.max_harmonics; h++)
                {
                    set_harmonic_phase(h, (L[h] == "Inverted"));
                }
            }
            catch (Exception Ex)
            {
                error_text = "Error executing phase CSV : " + Ex.Message;
            }

            return error_text;
        }

        public Double CalcLevelScaling()
        {
            if ((_CGS.Table_waveform_set.CurrentRow.RowIsEmpty == true) || (waveform_set_id == 0)) { return (Double)1.0; }

            if (flag_level_scaling_recalc_due == false)
            { return last_calculated_level_scaling; }


            Int32[] sample = new Int32[DA_Spectral.wavetable_samples]; //Note use of int. This is to mirror the DSP chip code.
            for (int i = 0; i < DA_Spectral.wavetable_samples; i++) { sample[i] = 0; }

            double ar_step;
            Double ar;

            for (int h_index = 0; h_index < DA_Spectral.max_harmonics; h_index++)
            {
                ar_step = ((Math.PI * 2) / DA_Spectral.wavetable_samples) * (h_index + 1);
                waveform_harmonicRow R = _CGS.Table_waveform_harmonic.GetRow(waveform_set_id, note_sector_id, intensity_layer_id, waveform_id, h_index);
                if (R != null)
                {
                    if (R.level > 0)
                    {
                        ar = 0;
                        for (int i = 0; i < DA_Spectral.wavetable_samples; i++)
                        {
                            sample[i] += (int)Math.Floor(Math.Sin(ar) * 32767.0 * (R.level / 65536.0));
                            ar += ar_step;
                        }
                    }
                }
            }

            Double ls;
            Double max_abs = 0;
            for (int i = 0; i < 2048; i++) { if (Math.Abs(sample[i]) > max_abs) { max_abs = Math.Abs(sample[i]); } }

            if (max_abs == 0)
            { ls = (Double)(1.0); }
            else
            {
                ls = (Double)(32767.0 / max_abs);
            }

            ls = ls / 256.0; //The app works with 16 bit levels but the module works with 8 bit levels.
            last_calculated_level_scaling = ls;
            flag_level_scaling_recalc_due = false;
            return ls;
        }

        public void morph_harmonics(c_from_to_range r)
        {
            double from_level = get_harmonic_level(r.from_id);
            double to_level = get_harmonic_level(r.to_id);
            double step_value = (to_level - from_level) / (r.to_id - r.from_id);
            for (int h = r.from_id; h <= r.to_id;h++)
            {
                set_harmonic_level(h,Convert.ToInt32(from_level + (step_value * (h - r.from_id))));
            }
        }

        #endregion

        #region "========================= METHODS - IMPORT HARMONICS =============================="
        public string input_harmonic_levels(ref string input)
        {
            string error_text = "";
            System.Drawing.Size size = new System.Drawing.Size(800, 80);
            Form inputBox = new Form();

            inputBox.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedDialog;
            inputBox.ClientSize = size;
            inputBox.Text = "Input harmonic levels as comma-separated list";
            inputBox.StartPosition = FormStartPosition.CenterScreen;
            inputBox.FormBorderStyle = FormBorderStyle.FixedSingle;
            inputBox.ControlBox = false;
            inputBox.Font = new Font("Arial", 8, FontStyle.Regular);

            System.Windows.Forms.TextBox textBox = new TextBox();
            textBox.Size = new System.Drawing.Size(size.Width - 10, 25);
            textBox.Location = new System.Drawing.Point(5, 5);
            textBox.Text = input;
            inputBox.Controls.Add(textBox);

            Button okButton = new Button();
            okButton.DialogResult = System.Windows.Forms.DialogResult.OK;
            okButton.Name = "okButton";
            okButton.Size = new System.Drawing.Size(75, 30);
            okButton.Text = "&OK";
            okButton.Location = new System.Drawing.Point(size.Width - 80 - 80, 39);
            inputBox.Controls.Add(okButton);

            Button cancelButton = new Button();
            cancelButton.DialogResult = System.Windows.Forms.DialogResult.Cancel;
            cancelButton.Name = "cancelButton";
            cancelButton.Size = new System.Drawing.Size(75, 30);
            cancelButton.Text = "&Cancel";
            cancelButton.Location = new System.Drawing.Point(size.Width - 80, 39);
            inputBox.Controls.Add(cancelButton);

            inputBox.AcceptButton = okButton;
            inputBox.CancelButton = cancelButton;

            DialogResult result = inputBox.ShowDialog();
            input = textBox.Text;
            if (result == DialogResult.OK)
            {
                error_text = set_harmonics_from_csv_list(input);
            }
            return error_text;
        }
        #endregion

        #region "========================= METHODS - WAVESHAPING ========================================="
        public void set_harmonic_level_shape_from_level_shape_type(harmonic_level_shape_types s)
        {
            switch (s)
            {
                case harmonic_level_shape_types.all_harmonics_max:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        set_harmonic_level(h,65535);
                    }
                    break;
                case harmonic_level_shape_types.all_harmonics_zero:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        set_harmonic_level(h, 0);
                    }
                    break;
                case harmonic_level_shape_types.exponential_all:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        set_harmonic_level(h, ((int)(((max_harmonics - h) / 32.0) * 65535 * 1.0 / (h + 1)))); 
                    }
                    break;
                case harmonic_level_shape_types.exponential_odd:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        if (((h + 1) % 2) == 0)
                        { set_harmonic_level(h, 0); }
                        else
                        {
                            set_harmonic_level(h, ((int)(((max_harmonics - h) / 32.0) * 65535 * 1.0 / (h + 1))));
                        }
                    }
                    break;
                case harmonic_level_shape_types.linearly_descreasing:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        set_harmonic_level(h, ((int)(65535 * 1.0 * ((max_harmonics - h) / 32.0))));
                    }
                    break;
            }
        }

        public void set_harmonic_level_shape_from_instrument_type(instrument_types i)
        {
            switch (i)
            {
                case instrument_types.cello:
                    clear_all_harmonic_levels();
                    set_harmonic_level(0, 65530);
                    set_harmonic_level(1, 39321);
                    set_harmonic_level(2, 39321);
                    set_harmonic_level(3, 45875);
                    set_harmonic_level(4, 29491);
                    set_harmonic_level(5, 13107);
                    set_harmonic_level(6, 29491);
                    set_harmonic_level(7, 7864);

                    break;
                case instrument_types.clarinet:
                    clear_all_harmonic_levels();
                    set_harmonic_level(0, 30000);
                    set_harmonic_level(1, 40000);
                    set_harmonic_level(2, 50000);
                    set_harmonic_level(3, 45000);
                    set_harmonic_level(4, 65530);
                    set_harmonic_level(5, 35000);
                    set_harmonic_level(6, 20000);
                    set_harmonic_level(7, 22000);
                    set_harmonic_level(8, 24000);
                    set_harmonic_level(9, 30000);
                    set_harmonic_level(10, 29000);
                    set_harmonic_level(11, 29500);
                    set_harmonic_level(12, 29600);
                    set_harmonic_level(13, 32000);
                    set_harmonic_level(14, 29000);
                    set_harmonic_level(15, 22000);
                    set_harmonic_level(16, 18000);
                    set_harmonic_level(17, 14000);
                    set_harmonic_level(18, 10000);
                    set_harmonic_level(19, 5000);
                    set_harmonic_level(20, 200);

                    break;
                case instrument_types.flute:
                    clear_all_harmonic_levels();
                    set_harmonic_level(0,65530);
                    set_harmonic_level(1,65530);
                    set_harmonic_level(2,7864);
                    set_harmonic_level(3,14417);
                    set_harmonic_level(4,1796);
                    set_harmonic_level(5,9000);
                    set_harmonic_level(6,9000);
                    set_harmonic_level(7,9000);
                    break;
                case instrument_types.guitar:
                    clear_all_harmonic_levels();
                    set_harmonic_level(0, 65535);
                    set_harmonic_level(1, 46670);
                    set_harmonic_level(2, 65535);
                    set_harmonic_level(3, 28420);
                    set_harmonic_level(4, 43490);
                    set_harmonic_level(5, 30565);
                    set_harmonic_level(6, 31002);
                    set_harmonic_level(7, 8771);
                    set_harmonic_level(8, 25950);
                    set_harmonic_level(9, 24365);
                    set_harmonic_level(10, 19560);
                    set_harmonic_level(11, 7424);
                    set_harmonic_level(12, 16224);
                    set_harmonic_level(13, 29269);
                    set_harmonic_level(14, 17853);
                    set_harmonic_level(15, 4216);
                    set_harmonic_level(16, 11802);
                    set_harmonic_level(17, 21713);
                    set_harmonic_level(18, 19213);
                    set_harmonic_level(19, 3898);
                    set_harmonic_level(20, 7641);
                    set_harmonic_level(21, 15594);
                    set_harmonic_level(22, 5994);
                    set_harmonic_level(23, 7424);
                    set_harmonic_level(24, 9007);
                    set_harmonic_level(25, 8083);
                    set_harmonic_level(26, 2883);
                    set_harmonic_level(27, 262);
                    set_harmonic_level(28, 0);
                    set_harmonic_level(29, 0);
                    set_harmonic_level(30, 0);
                    set_harmonic_level(31, 0);

                    break;
                case instrument_types.muted_guitar:
                    set_harmonic_level(0, 65535);
                    set_harmonic_level(1, 14741);
                    set_harmonic_level(2, 30303);
                    set_harmonic_level(3, 2929);
                    set_harmonic_level(4, 7962);
                    set_harmonic_level(5, 1337);
                    set_harmonic_level(6, 1559);
                    set_harmonic_level(7, 0);
                    set_harmonic_level(8, 618);
                    set_harmonic_level(9, 258);
                    set_harmonic_level(10, 2);
                    set_harmonic_level(11, 0);
                    set_harmonic_level(12, 53);
                    set_harmonic_level(13, 693);
                    set_harmonic_level(14, 1232);
                    set_harmonic_level(15, 0);
                    set_harmonic_level(16, 136);
                    set_harmonic_level(17, 77);
                    set_harmonic_level(18, 0);
                    set_harmonic_level(19, 0);
                    set_harmonic_level(20, 0);
                    set_harmonic_level(21, 0);
                    set_harmonic_level(22, 0);
                    set_harmonic_level(23, 0);
                    set_harmonic_level(24, 0);
                    set_harmonic_level(25, 0);
                    set_harmonic_level(26, 0);
                    set_harmonic_level(27, 0);
                    set_harmonic_level(28, 0);
                    set_harmonic_level(29, 0);
                    set_harmonic_level(30, 0);
                    set_harmonic_level(31, 0);

                    break;
                case instrument_types.trumpet:
                    clear_all_harmonic_levels();

                    break;
                case instrument_types.violin:
                    clear_all_harmonic_levels();
                    set_harmonic_level(0, 65530);
                    set_harmonic_level(1, 39321);
                    set_harmonic_level(2, 39321);
                    set_harmonic_level(3, 45875);
                    set_harmonic_level(4, 29491);
                    set_harmonic_level(5, 13107);
                    set_harmonic_level(6, 29491);
                    set_harmonic_level(7, 7864);

                    break;
            }
        }

        public void set_harmonic_level_shape_from_waveform_shape(waveform_shapes s)
        {
            switch (s)
            {
                case waveform_shapes.saw:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        set_harmonic_level(h, (int)(65535 / (h + 1.0))); 
                    }
                    break;
                case waveform_shapes.square:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        if (((h + 1) % 2) == 0)
                        { set_harmonic_level(h, 0); }
                        else
                        { set_harmonic_level(h, (int)(65535 / (h + 1.0))); }
                    }
                    break;

                case waveform_shapes.triangle:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        if (((h + 1) % 2) == 0)
                        { set_harmonic_level(h, 0); }
                        else
                        { set_harmonic_level(h, (int)(65535 * 1.0 / ((h + 1.0) * (h + 1.0)))); }
                    }
                    break;
            }
        }

        public void set_harmonic_phase_shape_from_phase_shape_type(harmonic_phase_shape_types s)
        {
            switch (s)
            {
                case harmonic_phase_shape_types.all_in_phase:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        set_harmonic_phase(h,false);
                    }
                    break;
                case harmonic_phase_shape_types.invert_3711:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        if (((h + 1) % 4) == 0)
                        { set_harmonic_phase(h, true); }
                        else
                        { set_harmonic_phase(h, false); }
                    }
                    break;
                case harmonic_phase_shape_types.invert_even:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        if ((h % 2) == 0)
                        { set_harmonic_phase(h, true); }
                        else
                        { set_harmonic_phase(h, false); }
                    }
                    break;
                case harmonic_phase_shape_types.invert_odd:
                    for (int h = 0; h < max_harmonics; h++)
                    {
                        if ((h % 2) == 0)
                        { set_harmonic_phase(h, false); }
                        else
                        { set_harmonic_phase(h, true); }
                    }
                    break;

            }
        }

        public void set_harmonic_phase_shape_from_instrument_type(instrument_types i)
        {
            //Not properly implemented
            switch (i)
            {
                case instrument_types.cello:
                    clear_all_harmonic_phases();
                    break;
                case instrument_types.clarinet:
                    clear_all_harmonic_phases();
                    break;
                case instrument_types.flute:
                    clear_all_harmonic_phases();
                    break;
                case instrument_types.guitar:
                    clear_all_harmonic_phases();
                    break;
                case instrument_types.muted_guitar:
                    clear_all_harmonic_phases();
                    break;
                case instrument_types.trumpet:
                    clear_all_harmonic_phases();
                    break;
                case instrument_types.violin:
                    clear_all_harmonic_phases();
                    break;
            }
        }

        public void set_harmonic_phase_shape_from_waveform_shape(waveform_shapes s)
        {
            switch(s)
            {
                case waveform_shapes.saw:
                    set_harmonic_phase_shape_from_phase_shape_type(harmonic_phase_shape_types.invert_even);
                    break;
                case waveform_shapes.square:
                    set_harmonic_phase_shape_from_phase_shape_type(harmonic_phase_shape_types.all_in_phase);
                    break;
                case waveform_shapes.triangle:
                    set_harmonic_phase_shape_from_phase_shape_type(harmonic_phase_shape_types.invert_even);
                    break;
            }
        }

        #endregion

        #region "============================ METHODS 3D ==========================================="
        public Double[] get_waveform_array()
        {
            Double AngleStep = (Double)((Math.PI * 2 * cycles) / waveform_plot_width);
            Double AngleRad;
            Double SinAR;

            Double[] waveform_signal = new Double[waveform_plot_width];

            for (int h_index = 0; h_index < DA_Spectral.max_harmonics; h_index++)
            {
                for (int i = 0; i < waveform_plot_width; i++)
                {
                    AngleRad = AngleStep * i * (h_index + 1);
                    if (get_harmonic_phase(h_index)) { SinAR = Math.Sin(AngleRad + Math.PI); } else { SinAR = Math.Sin(AngleRad); }
                    waveform_signal[i] += SinAR * get_harmonic_level(h_index); ;
                }
            }

            return waveform_signal;
        }

        public PolyLine3D get_3d_waveform_polyline(Pen waveform_pen, Pen highlighted_waveform_pen, double waveform_scaling,int z_index)
        {
            const int z_step = 100;
            Single Z;
            Double[] waveform_signal = get_waveform_array();

            PolyLine3D wp3d = new PolyLine3D();
            wp3d.LinePen = waveform_pen;
            wp3d.HighlightedLinePen = highlighted_waveform_pen;

            Z = z_step * (z_index - 2);
            for (int i = 0; i < waveform_plot_width; i++)
            {
                Point3D p = new Point3D(i - (waveform_plot_width / 2), (Single)(waveform_signal[i] * waveform_scaling), Z);
                wp3d.PointList.Add(p);
            }

            return wp3d;
        }

        public PolyLine3D get_3d_harmonic_polyline(Pen harmonic_pen, Pen highlighted_harmonic_pen,int z_index)
        {
            const int z_step = 100;
            const int column_width = 10;
            float harmonic_scaling = (float)(2 / 655.35);
            Single Z;

            PolyLine3D wp3d = new PolyLine3D();
            wp3d.LinePen = harmonic_pen;
            wp3d.HighlightedLinePen = highlighted_harmonic_pen;

            Z = z_step * (z_index - 2);


            wp3d.PointList.Add(new Point3D(-(harmonic_plot_width / 2), harmonic_y_offset, Z));

            for (int h_index = 0; h_index < DA_Spectral.max_harmonics; h_index++)
            {
                wp3d.PointList.Add(new Point3D(25 * h_index - (harmonic_plot_width / 2), harmonic_y_offset, Z));
                wp3d.PointList.Add(new Point3D(25 * h_index - (harmonic_plot_width / 2), -(Single)(get_harmonic_level(h_index) * harmonic_scaling) + harmonic_y_offset, Z));
                wp3d.PointList.Add(new Point3D(25 * h_index - (harmonic_plot_width / 2) + column_width, -(Single)(get_harmonic_level(h_index) * harmonic_scaling) + harmonic_y_offset, Z));
                wp3d.PointList.Add(new Point3D(25 * h_index - (harmonic_plot_width / 2) + column_width, harmonic_y_offset, Z));
            }

            return wp3d;
        }


        #endregion
    }
}
